<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Admin
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

/**
 * Register a new admin page, providing content and corresponding menu item for the CPT Archive Settings page.
 *
 * @package Genesis\Admin
 */
class Genesis_Admin_CPT_Archive_Settings extends Genesis_Admin_Boxes {

	/**
	 * Post type object.
	 *
	 * @var \stdClass
	 */
	protected $post_type;

	/**
	 * Layout selector enabled.
	 *
	 * @var bool
	 */
	protected $layout_enabled;

	/**
	 * Create an archive settings admin menu item and settings page for relevant custom post types.
	 *
	 * @since 2.0.0
	 *
	 * @param WP_Post_Type $post_type The post type object.
	 */
	public function __construct( $post_type ) {
		$this->post_type = $post_type;
		$this->help_base = GENESIS_VIEWS_DIR . '/help/cpt-archive-';

		/**
		 * Filter the enable CPT archive layout setting.
		 *
		 * @since 2.5.0
		 *
		 * @param bool $enable_layout Enable CPT archive layout setting. Default true.
		 */
		$this->layout_enabled = apply_filters( "genesis_cpt_archive_layout_setting_enable-{$this->post_type->name}", true );

		$page_id = 'genesis-cpt-archive-' . $this->post_type->name;

		$menu_ops = array(
			'submenu' => array(
				'parent_slug' => 'edit.php?post_type=' . $this->post_type->name,
				'page_title'  => apply_filters( 'genesis_cpt_archive_settings_page_label', __( 'Archive Settings', 'genesis' ) ),
				'menu_title'  => apply_filters( 'genesis_cpt_archive_settings_menu_label', __( 'Archive Settings', 'genesis' ) ),
				'capability'  => apply_filters( "genesis_cpt_archive_settings_capability_{$this->post_type->name}", 'manage_options' ),
			),
		);

		// Handle non-top-level CPT menu items.
		if ( is_string( $this->post_type->show_in_menu ) ) {
			$menu_ops['submenu']['parent_slug']   = $this->post_type->show_in_menu;
			$menu_ops['submenu']['menu_title']    = apply_filters( 'genesis_cpt_archive_settings_label', $this->post_type->labels->name . ' ' . __( 'Archive', 'genesis' ) );
			$menu_ops['submenu']['menu_position'] = $this->post_type->menu_position;
		}

		$page_ops = array(); // Use defaults.

		$settings_field = GENESIS_CPT_ARCHIVE_SETTINGS_FIELD_PREFIX . $this->post_type->name;

		$default_settings = apply_filters(
			'genesis_cpt_archive_settings_defaults',
			array(
				'headline'    => '',
				'intro_text'  => '',
				'doctitle'    => '',
				'description' => '',
				'keywords'    => '',
				'layout'      => '',
				'body_class'  => '',
				'noindex'     => 0,
				'nofollow'    => 0,
				'noarchive'   => 0,
			),
			$this->post_type->name
		);

		$this->create( $page_id, $menu_ops, $page_ops, $settings_field, $default_settings );

		add_action( 'genesis_settings_sanitizer_init', array( $this, 'sanitizer_filters' ) );
	}

	/**
	 * Register each of the settings with a sanitization filter type.
	 *
	 * @since 2.0.0
	 *
	 * @see \Genesis_Settings_Sanitizer::add_filter()
	 */
	public function sanitizer_filters() {

		genesis_add_option_filter(
			'no_html',
			$this->settings_field,
			array(
				'headline',
				'doctitle',
				'description',
				'keywords',
				'body_class',
				'layout',
			)
		);
		genesis_add_option_filter(
			'unfiltered_or_safe_html',
			$this->settings_field,
			array(
				'intro_text',
			)
		);
		genesis_add_option_filter(
			'one_zero',
			$this->settings_field,
			array(
				'noindex',
				'nofollow',
				'noarchive',
			)
		);
	}

	/**
	 * Register meta boxes on the CPT Archive pages.
	 *
	 * Some of the meta box additions are dependent on certain theme support or user capabilities.
	 *
	 * The 'genesis_cpt_archives_settings_metaboxes' action hook is called at the end of this function.
	 *
	 * @since 2.0.0
	 */
	public function metaboxes() {

		$this->add_meta_box( 'genesis-cpt-archives-settings', __( 'Archive Settings', 'genesis' ) );

		if ( ! genesis_seo_disabled() ) {
			$this->add_meta_box( 'genesis-cpt-archives-seo-settings', __( 'SEO Settings', 'genesis' ) );
		}

		if ( genesis_has_multiple_layouts() ) {
			$this->add_meta_box( 'genesis-cpt-archives-layout-settings', __( 'Layout Settings', 'genesis' ) );
		}

		/**
		 * Fires after CPT archive settings meta boxes have been added.
		 *
		 * @since 2.0.0
		 *
		 * @param string $pagehook Page hook for the CPT archive settings page.
		 */
		do_action( 'genesis_cpt_archives_settings_metaboxes', $this->pagehook );

	}

	/**
	 * Add contextual help content for the archive settings page.
	 *
	 * @since 2.0.0
	 */
	public function help() {

		$this->add_help_tab( 'archive', __( 'Archive Settings', 'genesis' ) );
		$this->add_help_tab( 'seo', __( 'SEO Settings', 'genesis' ) );
		$this->add_help_tab( 'layout', __( 'Layout Settings', 'genesis' ) );

	}
}
