<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Admin
 * @author  StudioPress
 * @license GPL-2.0+
 * @link    http://my.studiopress.com/themes/genesis/
 */

/**
 * Base class for Customizer classes in Genesis.
 */
abstract class Genesis_Customizer_Base {

	/**
	 * Define defaults, call the `register` method, add CSS to head.
	 *
	 * @since 2.1.0
	 */
	public function __construct() {

		// Register new customizer elements.
		if ( method_exists( $this, 'register' ) ) {
			add_action( 'customize_register', array( $this, 'register' ), 15 );
		} else {
			_doing_it_wrong( 'Genesis_Customizer_Base', __( 'When extending Genesis_Customizer_Base, you must create a register method.', 'genesis' ), '2.1.0' );
		}

		// Customizer scripts.
		if ( method_exists( $this, 'scripts' ) ) {
			add_action( 'customize_preview_init', 'scripts' );
		}

	}

	/**
	 * Get field name attribute value.
	 *
	 * @since 2.1.0
	 *
	 * @param string $name Option name.
	 * @return string Option name as key of settings field.
	 */
	protected function get_field_name( $name ) {
		return sprintf( '%s[%s]', $this->settings_field, $name );
	}

	/**
	 * Get field ID attribute value.
	 *
	 * @since 2.1.0
	 *
	 * @param string $id Option ID.
	 * @return string Option ID as key of settings field.
	 */
	protected function get_field_id( $id ) {
		return sprintf( '%s[%s]', $this->settings_field, $id );
	}

	/**
	 * Get field value.
	 *
	 * @since 2.1.0
	 *
	 * @param string $key Option key.
	 * @return mixed Field value.
	 */
	protected function get_field_value( $key ) {
		return genesis_get_option( $key, $this->settings_field );
	}

}

/**
 * Genesis Customizer.
 *
 * The class in which settings within Genesis core are added to the Customizer.
 */
class Genesis_Customizer extends Genesis_Customizer_Base {

	/**
	 * Settings field.
	 *
	 * @since 2.1.0
	 *
	 * @var string Settings field.
	 */
	public $settings_field = 'genesis-settings';

	/**
	 * Register new Customizer elements.
	 *
	 * Actual registration of settings and controls are handled in private methods.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize WP_Customize_Manager instance.
	 */
	public function register( $wp_customize ) {

		if ( ! current_theme_supports( 'genesis-custom-header' ) && ! current_theme_supports( 'custom-header' ) ) {
			$this->blog_title( $wp_customize );
		}

		$this->color_scheme( $wp_customize );

		if ( genesis_has_multiple_layouts() ) {
			$this->layout( $wp_customize );
		}

		if ( current_theme_supports( 'genesis-breadcrumbs' ) ) {
			$this->breadcrumbs( $wp_customize );
		}

		$this->comments( $wp_customize );
		$this->archives( $wp_customize );

	}

	/**
	 * Register the site title setting and control.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize WP_Customize_Manager instance.
	 */
	private function blog_title( $wp_customize ) {
		$wp_customize->add_setting(
			$this->get_field_name( 'blog_title' ),
			array(
				'default' => '',
				'type'    => 'option',
			)
		);

		$wp_customize->add_control(
			'blog_title',
			array(
				'label'    => __( 'Use for site title/logo:', 'genesis' ),
				'section'  => 'title_tagline',
				'settings' => $this->get_field_name( 'blog_title' ),
				'type'     => 'select',
				'choices'  => array(
					'text'  => __( 'Dynamic Text', 'genesis' ),
					'image' => __( 'Image logo', 'genesis' ),
				),
			)
		);
	}

	/**
	 * Register the color scheme setting and control.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize `WP_Customize_Manager` instance.
	 * @return void Return early if the theme does not support `genesis-style-selector`.
	 */
	private function color_scheme( $wp_customize ) {

		if ( ! current_theme_supports( 'genesis-style-selector' ) ) {
			return;
		}

		$wp_customize->add_section(
			'genesis_color_scheme',
			array(
				'title'    => __( 'Color Scheme', 'genesis' ),
				'priority' => '158.80',
			)
		);

		$wp_customize->add_setting(
			$this->get_field_name( 'style_selection' ),
			array(
				'default' => '',
				'type'    => 'option',
			)
		);

		$genesis_style_support = get_theme_support( 'genesis-style-selector' );

		$wp_customize->add_control(
			'genesis_color_scheme',
			array(
				'label'    => __( 'Select Color Style', 'genesis' ),
				'section'  => 'genesis_color_scheme',
				'settings' => $this->get_field_name( 'style_selection' ),
				'type'     => 'select',
				'choices'  => array_merge(
					array( '' => __( 'Default', 'genesis' ) ),
					array_shift( $genesis_style_support )
				),
			)
		);

	}

	/**
	 * Register the layout selector setting and control.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize WP_Customize_Manager instance.
	 */
	private function layout( $wp_customize ) {

		$wp_customize->add_section(
			'genesis_layout',
			array(
				'title'    => __( 'Site Layout', 'genesis' ),
				'priority' => '158.81',
			)
		);

		$wp_customize->add_setting(
			$this->get_field_name( 'site_layout' ),
			array(
				'default' => genesis_get_default_layout(),
				'type'    => 'option',
			)
		);

		$wp_customize->add_control(
			'genesis_layout',
			array(
				'label'    => __( 'Select Default Layout', 'genesis' ),
				'section'  => 'genesis_layout',
				'settings' => $this->get_field_name( 'site_layout' ),
				'type'     => 'select',
				'choices'  => genesis_get_layouts_for_customizer(),
			)
		);

	}

	/**
	 * Register the breadcrumbs settings and controls.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize WP_Customize_Manager instance.
	 */
	private function breadcrumbs( $wp_customize ) {

		$wp_customize->add_section(
			'genesis_breadcrumbs',
			array(
				'title'    => __( 'Breadcrumbs', 'genesis' ),
				'priority' => '158.82',
			)
		);

		$settings = array(
			'breadcrumb_home'       => __( 'Breadcrumbs on Homepage', 'genesis' ),
			'breadcrumb_front_page' => __( 'Breadcrumbs on Front Page', 'genesis' ),
			'breadcrumb_posts_page' => __( 'Breadcrumbs on Posts Page', 'genesis' ),
			'breadcrumb_single'     => __( 'Breadcrumbs on Single Posts', 'genesis' ),
			'breadcrumb_page'       => __( 'Breadcrumbs on Pages', 'genesis' ),
			'breadcrumb_archive'    => __( 'Breadcrumbs on Archives', 'genesis' ),
			'breadcrumb_404'        => __( 'Breadcrumbs on 404 Page', 'genesis' ),
			'breadcrumb_attachment' => __( 'Breadcrumbs on Attachment/Media', 'genesis' ),
		);

		$priority = 1;

		foreach ( $settings as $setting => $label ) {

			if ( 'breadcrumb_front_page' == $setting || 'breadcrumb_posts_page' == $setting ) {
				if ( 'page' !== get_option( 'show_on_front' ) ) {
					continue;
				}
			}

			if ( 'breadcrumb_home' == $setting && 'page' === get_option( 'show_on_front' ) ) {
				continue;
			}

			$wp_customize->add_setting(
				$this->get_field_name( $setting ),
				array(
					'default' => 0,
					'type'    => 'option',
				)
			);

			$wp_customize->add_control(
				'genesis_' . $setting,
				array(
					'priority' => $priority,
					'label'    => $label,
					'section'  => 'genesis_breadcrumbs',
					'settings' => $this->get_field_name( $setting ),
					'type'     => 'checkbox',
				)
			);

			$priority++;

		}

	}

	/**
	 * Register the comments and trackbacks settings and controls.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize WP_Customize_Manager instance.
	 */
	private function comments( $wp_customize ) {

		$wp_customize->add_section(
			'genesis_comments',
			array(
				'title'    => __( 'Comments and Trackbacks', 'genesis' ),
				'priority' => '158.83',
			)
		);

		$settings = array(
			'comments_posts'   => __( 'Enable Comments on Posts?', 'genesis' ),
			'comments_pages'   => __( 'Enable Comments on Pages?', 'genesis' ),
			'trackbacks_posts' => __( 'Enable Trackbacks on Posts?', 'genesis' ),
			'trackbacks_pages' => __( 'Enable Trackbacks on Pages?', 'genesis' ),
		);

		foreach ( $settings as $setting => $label ) {

			$wp_customize->add_setting(
				$this->get_field_name( $setting ),
				array(
					'default' => 0,
					'type'    => 'option',
				)
			);

			$wp_customize->add_control(
				'genesis_' . $setting,
				array(
					'label'    => $label,
					'section'  => 'genesis_comments',
					'settings' => $this->get_field_name( $setting ),
					'type'     => 'checkbox',
				)
			);

		}

	}

	/**
	 * Register the archives settings and controls.
	 *
	 * @since 2.1.0
	 *
	 * @param WP_Customize_Manager $wp_customize WP_Customize_Manager instance.
	 */
	private function archives( $wp_customize ) {

		$wp_customize->add_section(
			'genesis_archives',
			array(
				'title'       => __( 'Content Archives', 'genesis' ),
				'description' => __( 'These options will affect any blog listings page, including archive, author, blog, category, search, and tag pages.', 'genesis' ),
				'priority'    => '158.84',
			)
		);

		// Setting key and default value array.
		$settings = array(
			'content_archive'           => 'full',
			'content_archive_limit'     => '',
			'content_archive_thumbnail' => 0,
			'image_size'                => '',
			'image_alignment'           => 'alignleft',
			'posts_nav'                 => 'numeric',
		);

		foreach ( $settings as $setting => $default ) {

			$wp_customize->add_setting(
				$this->get_field_name( $setting ),
				array(
					'default' => $default,
					'type'    => 'option',
				)
			);

		}

		/**
		 * Filter available archive display options.
		 *
		 * @since 2.5.0
		 *
		 * @param array $args {
		 *     Contains archive display options.
		 *     @type string full     Entry content title.
		 *     @type string excerpts Entry excerpts title.
		 * }
		 */
		$archive_display = apply_filters(
			'genesis_archive_display_options',
			array(
				'full'     => __( 'Entry content', 'genesis' ),
				'excerpts' => __( 'Entry excerpts', 'genesis' ),
			)
		);

		$wp_customize->add_control(
			'genesis_content_archive',
			array(
				'priority' => 1,
				'label'    => __( 'Select one of the following', 'genesis' ),
				'section'  => 'genesis_archives',
				'settings' => $this->get_field_name( 'content_archive' ),
				'type'     => 'select',
				'choices'  => $archive_display,
			)
		);

		$wp_customize->add_control(
			'genesis_content_archive_limit',
			array(
				'label'    => __( 'Limit content to how many characters? (0 for no limit)', 'genesis' ),
				'section'  => 'genesis_archives',
				'settings' => $this->get_field_name( 'content_archive_limit' ),
			)
		);

		$wp_customize->add_control(
			'genesis_content_archive_thumbnail',
			array(
				'label'    => __( 'Display the featured image?', 'genesis' ),
				'section'  => 'genesis_archives',
				'settings' => $this->get_field_name( 'content_archive_thumbnail' ),
				'type'     => 'checkbox',
			)
		);

		$wp_customize->add_control(
			'genesis_image_size',
			array(
				'label'    => __( 'Featured image size', 'genesis' ),
				'section'  => 'genesis_archives',
				'settings' => $this->get_field_name( 'image_size' ),
				'type'     => 'select',
				'choices'  => genesis_get_image_sizes_for_customizer(),
			)
		);

		$wp_customize->add_control(
			'genesis_image_alignment',
			array(
				'label'    => __( 'Featured image alignment', 'genesis' ),
				'section'  => 'genesis_archives',
				'settings' => $this->get_field_name( 'image_alignment' ),
				'type'     => 'select',
				'choices'  => array(
					''           => __( '- None -', 'genesis' ),
					'alignleft'  => __( 'Left', 'genesis' ),
					'alignright' => __( 'Right', 'genesis' ),
					'aligncenter' => __( 'Center', 'genesis' ),
				),
			)
		);

		$wp_customize->add_control(
			'genesis_posts_nav',
			array(
				'label'    => __( 'Entry Pagination Type', 'genesis' ),
				'section'  => 'genesis_archives',
				'settings' => $this->get_field_name( 'posts_nav' ),
				'type'     => 'select',
				'choices'  => array(
					'prev-next' => __( 'Previous / Next', 'genesis' ),
					'numeric'   => __( 'Numeric', 'genesis' ),
				),
			)
		);

	}

}

add_action( 'init', 'genesis_customizer_init' );
/**
 * Create a new instance of the Genesis Customizer class.
 *
 * @since 2.1.0
 */
function genesis_customizer_init() {
	new Genesis_Customizer;
}

