<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Contributors
 * @author  StudioPress
 * @license GPL-2.0+
 * @link    http://my.studiopress.com/themes/genesis/
 */

/**
 * Repository class for managing contributors to Genesis.
 *
 * @since 2.5.0
 *
 * @package Genesis\Contributors
 */
class Genesis_Contributors {

	/**
	 * Cache for all the contributor details.
	 *
	 * @var Genesis_Contributor[]
	 */
	protected $people;

	/**
	 * Initialise Genesis_Contributors object.
	 *
	 * Each person in `$people` should have the following properties:
	 *   name
	 *   url (full URL)
	 *   avatar (full path to image)
	 *   role (typically `contributor` or `lead-developer`)
	 *
	 * There are two shortcut properties:
	 *   twitter (just the twitter handle, no `@`, which gets converted into a full URL for the `url` key)
	 *   gravatar (just the hash, which gets converted into a full Gravatar URL for the `avatar` key)
	 *
	 * The shortcut properties are preferred, but if someone does not have Twitter, or a Gravatar URL,
	 * different sources can be provided in the `url and `avatar` keys.
	 *
	 * @param array $people Data set of people who have contributed to Genesis.
	 */
	public function __construct( array $people ) {
		$all = array();

		foreach ( $people as $key => $person ) {
			if ( ! isset( $person['role'] ) ) {
				$person['role'] = 'none';
			}
			if ( ! isset( $person['url'] ) && isset( $person['twitter'] ) ) {
				$person['url'] = 'https://twitter.com/' . $person['twitter'];
			}
			if ( ! isset( $person['avatar'] ) && isset( $person['gravatar'] ) ) {
				$person['avatar'] = 'https://0.gravatar.com/avatar/' . $person['gravatar'] . '?s=120';
			}
			$all[ $key ] = new Genesis_Contributor( $person['name'], $person['url'], $person['avatar'],
				$person['role'] );
		}

		$this->people = $all;
	}

	/**
	 * Get all people who have contributed.
	 *
	 * @since 2.5.0
	 *
	 * @return Genesis_Contributor[]
	 */
	public function find_all() {
		return $this->people;
	}

	/**
	 * Find all contributors with a specific role.
	 *
	 * @since 2.5.0
	 *
	 * @param string $role Role to find contributors by.
	 * @return Genesis_Contributor[]
	 */
	public function find_by_role( $role ) {
		$people = array();
		foreach ( $this->people as $key => $person ) {
			if ( $role === $person->get_role() ) {
				$people[ $key ] = $person;
			}
		}

		return $people;
	}

	/**
	 * Get all contributors, in a shuffled order.
	 *
	 * @since 2.5.0
	 *
	 * @return Genesis_Contributor[]
	 */
	public function find_contributors() {
		$contributors = $this->find_by_role( 'contributor' );
		shuffle( $contributors );

		return $contributors;
	}

	/**
	 * Get a single contributor by their ID.
	 *
	 * The ID is typically the full name, lowercase, no spaces i.e. `nathanrice`.
	 *
	 * @since 2.5.0
	 *
	 * @param string $id Contributor ID.
	 * @return Genesis_Contributor Person matching ID `$id`.
	 */
	public function find_by_id( $id ) {
		return $this->people[ $id ];
	}
}
